 /**
 *
 * \file        HPRFGW_RFComLog.c
 *
 * \brief       This file contains the RF communication log handling routines.		
 *
 * \detail		
 *
 * \author      Hazrat Shah	
 *
 * \date        08/24/2006
 * \warning	
 * \note
 * \review
 */

#include "hprfgw_config.h"
#include "hprfgw_rfDesc.h"
#include "hprfgw_rfComLog.h"

////////////////////////////////////////////////////////////////////////////////
//	DEFINITIONS	
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//	LOCAL DATA		
////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
//				
////////////////////////////////////////////////////////////////////////////////
#ifdef	RF_DBG_LOG	
	TYPE_DBG_TXRX_TRACE_REC RFDbgTrace;
#endif

////////////////////////////////////////////////////////////////////////////////
//	LOCAL FUNCTIONS 
///////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
//	GLOBAL FUNCTIONS 
//////////////////////////////////////////////////////////////////////////////

#ifdef	RF_MSG_ERR_LOG
/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function retrieves the device message errors from buffer    	 	 			   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    UINT8 index, UINT8 *type, UINT8 *subtype, UINT8 *req, UINT8 *initiator, UNIT8 *code			 
 *
 * \return		UINT16	res  		 			
 *
 * \retval		SUCCESS/FAILURE 											  
 */
UINT16 RFComLog_GetRFDeviceMsgError (UINT8 index, UINT8 *type, UINT8 *subtype, UINT8 *req, UINT8 *initiator, UINT8 *code)	{					
	UINT16	res = FAILURE;			
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);

	if (pdst->COMLog.MsgErr.in != pdst->COMLog.MsgErr.out)	{
		*type    = pdst->COMLog.MsgErr.Err[pdst->COMLog.MsgErr.out].type;
		*subtype = pdst->COMLog.MsgErr.Err[pdst->COMLog.MsgErr.out].subtype;
		*req     = pdst->COMLog.MsgErr.Err[pdst->COMLog.MsgErr.out].req;
		*initiator = pdst->COMLog.MsgErr.Err[pdst->COMLog.MsgErr.out].initiator;
		*code = pdst->COMLog.MsgErr.Err[pdst->COMLog.MsgErr.out].code;
		if (pdst->COMLog.MsgErr.out++ >= RF_DEVICE_MAX_MSG_ERRORS)	{
			pdst->COMLog.MsgErr.out = 0;	
		}
		res = SUCCESS;						
	}	

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function stores the device message error while either posting to stack or receiving packet confirmation from stack for this device    	 			   			           
 *
 * \detail		
 * \date        09/07/2006
 *
 * \param	    UINT8 index, UINT8 type, UINT8 subtype, UINT8 req, UINT8 initiator, UINT8 code			 
 *
 * \return		UINT16	res		 			
 *
 * \retval		SUCCESS, FAILURE  (returns SUCCESS if read successfully)											  
 */
UINT16 RFComLog_PutRFDeviceMsgError (UINT8 index, UINT8 type, UINT8 subtype, UINT8 req, UINT8 initiator, UINT8 code)	{				
	UINT16	res = FAILURE;			
	UINT8	in;	
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;
	
	in  = pdst->COMLog.MsgErr.in+1;
	if (in >= RF_DEVICE_MAX_MSG_ERRORS)	{	in = 0;		}
	 
	if (in != pdst->COMLog.MsgErr.out)	{
		pdst->COMLog.MsgErr.Err[pdst->COMLog.MsgErr.in].type	= type;
		pdst->COMLog.MsgErr.Err[pdst->COMLog.MsgErr.in].subtype = subtype;
		pdst->COMLog.MsgErr.Err[pdst->COMLog.MsgErr.in].req		= req;
		pdst->COMLog.MsgErr.Err[pdst->COMLog.MsgErr.in].initiator= initiator;
		pdst->COMLog.MsgErr.Err[pdst->COMLog.MsgErr.in].code = code;
		pdst->COMLog.MsgErr.in = in;
		res = SUCCESS;						
	}	

	return (res);
}
#endif

#ifdef COMM_LOG_PARS
/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function reads the tx totaltxframefailsincelastactivelinkcnt	 	 			   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    index			 
 *
 * \return		count 		 			
 *
 * \retval		Returns totaltxframefailsincelastactivelinkcnt count											  
 */
UINT16 RFComLog_GetRFDevicetotaltxframefailsincelastactivelinkcnt (UINT8 index)	{				
	UINT16	count = 0;			
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;

	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		count = pdst->COMLog.totaltxframefailsincelastactivelinkcnt;
	}

	return (count);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function clears the tx totaltxframefailsincelastactivelinkcnt	 	 			   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    UINT8 index			 
 *
 * \return		UINT16 result		 			
 *
 * \retval		SUCCESS, FAILURE  (returns SUCCESS if read successfully)											  
 */
UINT16 RFComLog_ClearRFDevicetotaltxframefailsincelastactivelinkcnt (UINT8 index)	{			
	UINT16	result = FAILURE;		
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;
	
	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		pdst->COMLog.totaltxframefailsincelastactivelinkcnt	= 0;
		result = SUCCESS;
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function clears the tx buffer over flow counters 	 	 			   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    index			 
 *
 * \return		result		 			
 *
 * \retval		SUCCESS, FAILURE  (returns SUCCESS if read successfully)											  
 */
UINT16 RFComLog_ClearRFDeviceTxBufOverFlowParameters (UINT8 index)	{			
	UINT16	result = FAILURE;		
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;
	
	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		pdst->COMLog.txbufoverflowflag	   = 0;
		pdst->COMLog.totaltxbufoverflowcnt = 0;
		result = SUCCESS;
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function clears the rx buffer over flow counters 	 	 			   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    index			 
 *
 * \return		result		 			
 *
 * \retval		SUCCESS, FAILURE  (returns SUCCESS if read successfully)											  
 */
UINT16 RFComLog_ClearRFDeviceRxBufOverFlowParameters (UINT8 index)	{		
	UINT16	result = FAILURE;		
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;
	
	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		pdst->COMLog.rxbufoverflowflag      = 0;
		pdst->COMLog.totalrxbufoverflowcnt  = 0;
		result = SUCCESS;
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function reads the txbuffer over flow counters 	 	 			   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    index, *pTxOVFflg, *pTotalTxOVFCntr		 
 *
 * \return		result		 			
 *
 * \retval		SUCCESS, FAILURE  (returns SUCCESS if read successfully)											  
 */
UINT16 RFComLog_GetRFDeviceTxBufOverFlowParameters (UINT8 index, UINT8 *pTxOVFflg, UINT32 *pTotalTxOVFCntr)	{		
	UINT16	result = FAILURE;		
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;

	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		*pTxOVFflg 		 = pdst->COMLog.txbufoverflowflag;
		*pTotalTxOVFCntr = pdst->COMLog.totaltxbufoverflowcnt;
		result = SUCCESS;
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function reads the rx buffer over flow counters 	 	 			   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    index, *pRxOVFflg, *pTotalRxOVFCntr		 
 *
 * \return		result		 			
 *
 * \retval		SUCCESS, FAILURE  (returns SUCCESS if read successfully)											  
 */
UINT16 RFComLog_GetRFDeviceRxBufOverFlowParameters (UINT8 index, UINT8 *pRxOVFflg, UINT32 *pTotalRxOVFCntr)	{		
	UINT16	result = FAILURE;		
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;
 	
	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		*pRxOVFflg 		 = pdst->COMLog.rxbufoverflowflag;
		*pTotalRxOVFCntr = pdst->COMLog.totalrxbufoverflowcnt;
		result = SUCCESS;
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the rf device link state to active	 	 			   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    index		 
 *
 * \return		result		 			
 *
 * \retval		SUCCESS, FAILURE 			
 */
UINT16 RFComLog_SetRFDeviceLinkStateToActive (UINT8 index)	{		
	UINT16	result = FAILURE;		
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;

	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		pdst->linkstate = RF_DEVICE_LINK_STATE_ACTIVE;	
		result = SUCCESS;
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function increment the link state count 	 	 			   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    index		 
 *
 * \return		result		 			
 *
 * \retval		SUCCESS, FAILURE 			
 */
UINT16 RFComLog_IncrementRFDeviceLinkStateCount (UINT8 index)	{		
	UINT16	result = FAILURE;		
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;

	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		pdst->linkstatecnt++;		
		result = SUCCESS;
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function increment the total number received frame count from this device 	 			   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    index		 
 *
 * \return		result		 			
 *
 * \retval		SUCCESS, FAILURE 			
 */
UINT16 RFComLog_IncrementRFDeviceTotalRxFrameCount (UINT8 index)	{		
	UINT16	result = FAILURE;		
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;

	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		pdst->COMLog.totalrxframecnt++;	
		result = SUCCESS;
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function logs the rf device repeat frames	 			   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    index		 
 *
 * \return		result		 			
 *
 * \retval		SUCCESS, FAILURE 			
 */
UINT16 RFComLog_RFDeviceRepeatFrameError (UINT8 index)	{		
	UINT16	result = FAILURE;		
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;

	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		pdst->COMLog.totalrxframerepeatcnt++;	
		result = SUCCESS;
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function logs the rf device sequence count error 			   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    index		 
 *
 * \return		result		 			
 *
 * \retval		SUCCESS, FAILURE 			
 */
UINT16 RFComLog_RFDeviceSequenceCountError (UINT8 index)	{		
	UINT16	result = FAILURE;		
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;
 	
	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		pdst->COMLog.rxframesequenceerrcnt++;	
		result = SUCCESS;
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function syncronizes the receiver sequence count			   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    index, sequencecount	 
 *
 * \return		result		 			
 *
 * \retval		SUCCESS, FAILURE 			
 */
UINT16 RFComLog_SyncRFDeviceReceiverSequenceCount(UINT8 index, UINT8 sequencecnt)	{	
	UINT16	result = FAILURE;		
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;

	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		pdst->rxsequence = sequencecnt;
		result = SUCCESS;
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the receiver frame sequence count		   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    index	 
 *
 * \return		sequence count	 			
 *
 * \retval		0..255			
 */
UINT8 RFComLog_GetRFDeviceReceiverSequenceCount (UINT8 index)	{	
	UINT8	sequencecnt = 0;	
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;
 	
	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		sequencecnt = pdst->rxsequence;
	}

	return (sequencecnt);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the device link state	   			           
 *
 * \detail		
 * \date        09/06/2006
 *
 * \param	    index	 
 *
 * \return		linkstate 			
 *
 * \retval		NOT CONNECT, ACTIVE, PARTIAL, FAIL, DISCONNECTED		
 */
UINT8 RFComLog_GetRFDeviceLinkState (UINT8 index)	{	
	UINT8	linkstate = RF_DEVICE_LINK_STATE_NOT_CONNECTED;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;
 	
	if (index < RF_DESCRIPTIVE_TABLE_SIZE)	{
		linkstate = pdst->linkstate;
	}

	return (linkstate);
}

#endif  

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		RF debug communication log param init 	   			           
 *
 * \detail		
 * \date        07/17/2007
 *
 * \param	    void	 
 *
 * \return		void 			
 *
 * \retval				
 */
void RFComLog_DbgLogInit (void)	{
#ifdef	RF_TXRX_DBG
	memset ((UINT8*)&RFDbgTrace, 0, sizeof(RFDbgTrace));
#endif
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		Log rf RF debug event   	   			           
 *
 * \detail		
 * \date        07/17/2007
 *
 * \param	    UINT8 evcode, UINT8 index, UINT8 type, UINT8	pkttype, UINT8	pkt, UINT32	time	 
 *
 * \return		INT32			
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFComLog_LogEvent (UINT8 evcode, UINT8 index, UINT8 type, UINT8	pkttype, UINT8	pkt, UINT32	pktcnt, UINT32	time)	{
#ifdef	RF_DBG_LOG
	RFDbgTrace.RFdbg[RFDbgTrace.rfdbgindex].evcode = evcode;
	RFDbgTrace.RFdbg[RFDbgTrace.rfdbgindex].index = index;
	RFDbgTrace.RFdbg[RFDbgTrace.rfdbgindex].type = type;
	RFDbgTrace.RFdbg[RFDbgTrace.rfdbgindex].pkttype = pkttype;
	RFDbgTrace.RFdbg[RFDbgTrace.rfdbgindex].pkt = pkt;
	RFDbgTrace.RFdbg[RFDbgTrace.rfdbgindex].pktcnt = pktcnt;
	RFDbgTrace.RFdbg[RFDbgTrace.rfdbgindex].time = time;
	if (++RFDbgTrace.rfdbgindex >= RF_DBG_TRACE_MAX_SIZE)	{	RFDbgTrace.rfdbgindex = 0;	}
#else

#endif
	return (SUCCESS);
}
	
/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		Retrieve comm log events  	   			           
 *
 * \detail		
 * \date        07/17/2007
 *
 * \param	    UINT8 *pevcode, UINT8 *pindex, UINT8 *ptype, UINT8	*ppkttype, UINT8 *ppkt, UINT32 *ppktcnt, UINT32 *ptime, UINT32 *ptblindex	 
 *
 * \return		INT32 			
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFComLog_RetreiveEvent (UINT8 *pevcode, UINT8 *pindex, UINT8 *ptype, UINT8	*ppkttype, UINT8 *ppkt, UINT32 *ppktcnt, UINT32 *ptime, UINT32 *ptblindex)	{
#ifdef	RF_DBG_LOG
	*pevcode = RFDbgTrace.RFdbg[*ptblindex].evcode;
	*pindex = RFDbgTrace.RFdbg[*ptblindex].index;
	*ptype = RFDbgTrace.RFdbg[*ptblindex].type;
	*ppkttype = RFDbgTrace.RFdbg[*ptblindex].pkttype;
	*ppkt = RFDbgTrace.RFdbg[*ptblindex].pkt;
	*ppktcnt = RFDbgTrace.RFdbg[*ptblindex].pktcnt;
	*ptime = RFDbgTrace.RFdbg[*ptblindex].time;
	if (++*ptblindex >= RF_DBG_TRACE_MAX_SIZE)	{	*ptblindex = 0;	}
#else
	//if (*ptblindex)	{	*ptblindex -= 1;	}	else	{	*ptblindex = (RF_DBG_TRACE_MAX_SIZE-1);	}		
	*pevcode = 0;	
	*pindex = 0;	
	*ptype = 0;
	*ppkttype = 0;
	*ppkt = 0;
	*ppktcnt = 0;
	*ptime = 0;
#endif
	return (SUCCESS);
}
	
/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		Retrieve rf debug log current index   	   			           
 *
 * \detail		
 * \date        07/17/2007
 *
 * \param	    UINT32 *pindex, UINT32 cnt 
 *
 * \return		INT32 			
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFComLog_RetreiveCurrentEventIndex (UINT32 *pindex, UINT32 cnt)	{
#ifdef	RF_DBG_LOG
	if (cnt >= RF_DBG_TRACE_MAX_SIZE)	{	cnt = RF_DBG_TRACE_MAX_SIZE;	}
	if (cnt > RFDbgTrace.rfdbgindex)	{
		*pindex = RF_DBG_TRACE_MAX_SIZE-(cnt-RFDbgTrace.rfdbgindex)+1; 
	}	else	{
		*pindex = RFDbgTrace.rfdbgindex-cnt;
	}
#else
	*pindex = 0;
#endif
	return (SUCCESS);
}
